# Define the input file names
# These files are expected to be in the same directory as the script.
$TeamFiles = @("Teams.csv", "MiLBTeams.csv")

# Define the old and new team names
$OldTeamName = "Guardians"
$NewTeamName = "Indians"
$YearThreshold = 2022

Write-Host "Starting CSV modification process with backup..."

foreach ($inputFile in $TeamFiles) {
    $currentDirectory = Get-Location
    $inputPath = Join-Path -Path $currentDirectory -ChildPath $inputFile
    $backupPath = "$($inputPath).BAK" # Path for the backup file

    Write-Host "`nProcessing file: $($inputFile)"

    # --- Step 1: Check if input file exists ---
    if (-not (Test-Path $inputPath)) {
        Write-Warning "Error: File not found at $($inputPath). Skipping this file."
        continue # Skip to the next file in the list
    }

    # --- Step 2: Create a backup of the original file ---
    try {
        if (Test-Path $backupPath) {
            # If a .BAK file already exists, remove it to ensure a fresh backup
            Remove-Item -LiteralPath $backupPath -Force
            Write-Host "Removed existing backup: $($backupPath)"
        }
        Rename-Item -LiteralPath $inputPath -NewName $backupPath -ErrorAction Stop
        Write-Host "Created backup: $($backupPath)"
    }
    catch {
        Write-Error "Failed to create backup for $($inputFile). Original file might still be there. Error: $($_.Exception.Message). Skipping this file."
        continue # Skip to the next file if backup fails
    }

    # --- Step 3: Process the (now backed-up) CSV and save with original name ---
    try {
        # Import the data from the backup file, as the original name is now the backup
        $data = Import-Csv -LiteralPath $backupPath

        $modifiedCount = 0

        # Iterate through each row and apply the condition and replacement
        foreach ($row in $data) {
            # Attempt to convert yearID to integer for comparison
            $currentYearId = [int]$null
            try {
                $currentYearId = [int]$row.yearID
            } catch {
                # This warning can be verbose for large files with many non-numeric yearIDs.
                # You might comment it out if you expect many, or refine the error handling.
                Write-Warning "Could not convert 'yearID' value '$($row.yearID)' to a number in a row. Skipping year check for this row."
                continue # Skip to next row if yearID is not a valid number
            }

            # Check the yearID condition
            if ($currentYearId -lt $YearThreshold) {
                # Iterate through all properties (columns) of the row
                foreach ($property in $row.PSObject.Properties) {
                    # Only process if the value is a string and contains the old team name
                    if ($property.Value -is [string] -and $property.Value -like "*$OldTeamName*") {
                        $originalValue = $property.Value
                        $property.Value = $property.Value -ireplace $OldTeamName, $NewTeamName
                        if ($property.Value -ne $originalValue) {
                            $modifiedCount++
                        }
                    }
                }
            }
        }

        # Export the modified data to the original file name
        $data | Export-Csv -LiteralPath $inputPath -NoTypeInformation

        Write-Host "Successfully modified and overwrote '$($inputFile)'. Modified $($modifiedCount) instances."
        Write-Host "Original backed up to: $($backupPath)"

    }
    catch {
        Write-Error "An unexpected error occurred while processing $($inputFile) (after backup). Error: $($_.Exception.Message)"
        Write-Error "Check $($backupPath) for the original data."
    }
}

Write-Host "`nCSV modification process complete."